#!/usr/bin/env python3
#
# Copyright (c) 2019 Poul-Henning Kamp <phk@phk.freebsd.dk>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

'''Decode KryoFlux streamfiles of Commodore 900 format floppies'''

import sys

GCR = {
	"01010":	0x00,
	"01011":	0x01,
	"10010":	0x02,
	"10011":	0x03,
	"01110":	0x04,
	"01111":	0x05,
	"10110":	0x06,
	"10111":	0x07,
	"01001":	0x08,
	"11001":	0x09,
	"11010":	0x0A,
	"11011":	0x0B,
	"01101":	0x0C,
	"11101":	0x0D,
	"11110":	0x0E,
	"10101":	0x0F,
}

def decode_gcr(b):
	''' Decode GCR to binary from ASCII [01]-string '''
	l = []
	i = 0
	while i < len(b) and b[i] == '1':
		i += 1
	while i < len(b) - 10:
		t = GCR.get(b[i:i+5])
		r = GCR.get(b[i+5:i+10])
		if t is None or r is None:
			break
		i += 10
		l.append((t<<4) | r)
	return i, l

class Track():
	''' Process one track-file in KryoFlux STREAM format '''

	def __init__(self, tr, fn):
		self.tr = tr
		self.fn = fn
		self.dt = []
		self.readfile()

	def readfile(self):
		''' Read KryoFlux Stream file, convert to list of delta-T '''
		b = open(self.fn, 'rb').read()
		i = 0
		while i < len(b):
			if b[i] == 0x0d:
				l = b[i + 2]
				l += b[i + 3] << 8
				i += 4 + l
			elif b[i] == 0x0a:
				i += 3
			elif b[i] < 0x08:
				d = (b[i] << 8) + b[i + 1]
				i += 2
				self.dt.append(d)
			elif b[i] >= 0x0e:
				d = b[i]
				i += 1
				self.dt.append(d)
			else:
				print("# ??? %02x %02x %02x %02x" % (b[i+0], b[i+1], b[i+2], b[i+3]))
				break

	def discriminate(self, lo, hi):
		''' Convert delta-T to binary bit-stream '''
		self.bits = []
		for i in self.dt:
			if i < lo:
				self.bits.append("")
			elif i < hi:
				self.bits.append("0")
			else:
				self.bits.append("00")
		self.bits = "1".join(self.bits)

	def interpret(self):
		''' Interpret bit-stream as track '''
		rval = [None] * 16
		part = self.bits.split('11111111111111111')
		hdr = None
		for i in part:
			if not i:
				continue
			j,l = decode_gcr(i)
			if not l:
				#print("?", len(l), l[:10])
				pass
			elif l[0] == 0x08:
				hdr = self.header(l)
			elif hdr and l[0] == 0x07 and len(l) >= 514:
				b = self.body(l)
				if b:
					rval[hdr[2]] = b
			else:
				#print("?", len(l), l[:10])
				pass
		return rval

	def header(self, b):
		''' Decode ID field '''
		x = 0
		h = b[:4]
		for i in h:
			x ^= i
		if not x:
			return h
		return None

	def body(self, b):
		''' Decode sector content '''
		x = 0
		for i in b[:514]:
			x ^= i
		if not x:
			return b[1:513]
		return None


def DoStreamFloppy(fn):
	print("FN", fn)
	fo = open(fn + ".bin", "wb")
	for tr in range(0, 77):
		if tr < 39:
			ns = 16
			d = 2
		elif tr < 53:
			ns = 15
			d = 5
		elif tr < 64:
			ns = 14
			d = 7
		else:
			ns = 13
			d = 10
		for side in (0, 1):
			t = Track(tr, '%s/track%02d.%d.raw' % (fn, tr, side))
			t.discriminate(65 + d, 105 + d + d)
			r = t.interpret()
			x = ""
			g = 0
			for i in r:
				if i is None:
					x += '-'
				else:
					fo.write(bytearray(i))
					x += '+'
					g += 1
			assert g == ns
			print("%2d" % tr, side, "%2d" % d, "%2d" % g, x)
			sys.stdout.flush()
	print()

for res in ("low", "high"):
	for v in (1, 2, 3):
		DoStreamFloppy('Vol_%d_%s_resolution' % (v, res))
