#!/usr/bin/env python3
#
# Copyright (c) 2019 Poul-Henning Kamp <phk@phk.freebsd.dk>
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

'''Analyse/Extract from Commodore900 (Floppy) Filesystem'''

import time
import hashlib

def canl(b, o):
	assert o + 4 <= len(b)
	x = b[o + 2]
	x |= b[o + 3] << 8
	x |= b[o + 0] << 16
	x |= b[o + 1] << 24
	return x

def n(b):
	x = 0
	for i in b:
		x <<= 8
		x |= i
	#print("n", b, x, "%x" % x)
	return x

class superblock(object):
	def __init__(self, fs):
		b1 = fs.bread(fs.SUPERI)
		b = bytearray()
		for i in range(0,len(b1),2):
			b.append(b1[i + 1])
			b.append(b1[i])

		p = 0

		self.s_isize = n(b[p:p+2])
		p += 2

		self.s_fssize = n(b[p:p+4])
		p += 4

		self.s_nfree = n(b[p:p+2])
		p += 2

		self.s_free = list()
		for i in range(0, fs.NICFREE):
			self.s_free.append(n(b[p:p+4]))
			p += 4

		self.s_ninode = n(b[p:p+2])
		p += 2

		self.s_inode = list()
		for i in range(0, fs.NICINOD):
			self.s_inode.append(n(b[p:p+2]))
			p += 2

		self.s_flock = b[p]
		p += 1

		self.s_ilock = b[p]
		p += 1

		self.s_fmod = b[p]
		p += 1

		self.s_ronly = b[p]
		p += 1

		self.s_time = n(b[p:p+4])
		p += 4

		self.s_tfree = n(b[p:p+4])
		p += 4

		self.s_tinode = n(b[p:p+2])
		p += 2

		self.s_m = n(b[p:p+2])
		p += 2

		self.s_n = n(b[p:p+2])
		p += 2

		self.s_fname = b[p:p+6]
		p += 6

		self.s_fpack = b[p:p+6]
		p += 6

		self.s_unique = n(b[p:p+4])
		p += 4

		if False:
			print(p, b[p:])
			for i in self.__dict__:
				print(i, "\t", self.__dict__[i])

class inode(object):
	def __init__(self, fs, ino):
		self.fs = fs
		self.i_ino = ino

	def init(self):
		ino = self.i_ino
		ino -= 1
		b = self.fs.bread((ino >> 3) + self.fs.INODEI)
		i = ino & 7
		b1 = b[64 * i: 64 * i + 64]
		b = bytearray()
		for i in range(0,len(b1),2):
			b.append(b1[i + 1])
			b.append(b1[i])

		p = 0

		self.i_mode = n(b[p:p+2])
		p += 2
		typ = self.i_mode & 0o170000
		#print("%o" % self.i_mode, "%o" % typ)

		self.i_nlink = n(b[p:p+2])
		p += 2

		self.i_uid = n(b[p:p+2])
		p += 2

		self.i_gid = n(b[p:p+2])
		p += 2

		if typ == 0o100000 or typ == 0o40000:
			self.i_size = n(b[p:p+4])
			p += 4
			p += 1
			self.i_addr = list()
			for i in range(0, 13):
				x = b1[p]
				x |= b1[p + 1] << 8
				x |= b1[p + 2] << 16
				self.i_addr.append(x)
				p += 3
		elif typ == 0o20000 or typ == 0o60000:
			p += 4
			self.i_dev = n(b[p:p+2])
			p += 2
		else:
			#print("TYP %o" % typ)
			#print(p, b1[p:52])
			#for i in self.__dict__:
			#	print(i, "\t", self.__dict__[i])
			self.xdata = b1[p:52]

		p = 52
		self.i_atime = n(b[p:p+4])
		p += 4
		self.i_mtime = n(b[p:p+4])
		p += 4
		self.i_ctime = n(b[p:p+4])
		p += 4

		if False:
			for i in self.__dict__:
				print(i, "\t", self.__dict__[i])
		if typ == 0o40000:
			self.dir = dict()
			d = self.get_data()
			for i in range(0, len(d), 16):
				inx = d[i] | (d[i + 1] << 8)
				s = ""
				for j in range(2, 16):
					if d[i + j] == 0:
						break
					s += "%c" % d[i + j]
				if inx != 0:
					self.dir[s] = (inx, s, self.fs.geti(inx))

	def get_bn(self, bn):
		if bn < 10:
			# Direct Block
			return self.i_addr[bn]
		bn -= 10

		if bn < 128:
			# Indirect Block
			d = self.fs.bread(self.i_addr[10])
			x = canl(d, 4 * bn)
			return x
		bn -= 128

		# Doubly Indirect Block
		d = self.fs.bread(self.i_addr[11])
		x = canl(d, bn >> 7)
		d = self.fs.bread(x)
		x = canl(d, 4 * (bn & 0x7f))
		return x
		

	def get_data(self):
		b = bytearray()
		nb = (self.i_size + self.fs.BSIZE - 1) >> 9
		l = self.i_size
		for bn in range(0, nb):
			bnx = self.get_bn(bn)
			#print(nb, bnx, self.i_addr)
			bx = self.fs.bread(bnx)
			if l < len(bx):
				bx = bx[:l]
			b += bx
			l -= len(bx)
		assert len(b) == self.i_size
		return b

	def render(self, pfx, lev, fo):
		typ = self.i_mode & 0o170000
		s = ""
		s += " ino=%d" % self.i_ino
		s += " typ=%o" % self.i_mode
		# hashlib.sha224(b"Nobody inspects the spammish repetition").hexdigest()
		if typ == 0o100000:
			s += " md5=" + hashlib.md5(self.get_data()).hexdigest()
		for i in self.__dict__:
			if i == "fs" or i == "dir" or i == "i_mode" or i == "i_ino":
				continue
			if i == "i_atime" or i == "i_mtime" or i == "i_ctime" or i == "i_addr":
				continue
			s += " " + i + "=" + str(self.__dict__[i])
		fo.write(lev + pfx + s + "\n")
		if typ == 0o40000:
			for j in self.dir:
				inx,i,ino = self.dir[j]
				if i != "." and i != "..":
					ino.render(pfx + "/" + i, lev + "\t", fo)
				if False and (ino.i_mode & 0o170000) == 0o100000:
					print("Found", j)
					fx = open("__" + j, "wb")
					fx.write(ino.get_data())
					fx.close()
		

class coherent_fs(object):
	def __init__(self, fn):
		print("---------------", fn)
		fi = open(fn, "rb")
		self.img = bytearray(fi.read())
		fi.close()

		# self.rootdir = coherent_fs_dir(2)

		# filsys.h: Block size
		self.BSIZE = 512
		# fblk.h: Number of free blocks in free list
		self.NICFREE = 64
		# filsys.h: Number of free in core inodes
		self.NICINOD = 100
		# filsys.h: Boot block index
		self.BOOTBI = 0
		# filsys.h: Super block index
		self.SUPERI = 1
		# filsys.h: Inode block index
		self.INODEI = 2
		# filsys.h: Bad block inode number
		self.BADFIN = 1
		# filsys.h:  Root inode number
		self.ROOTIN = 2

		self.icache = dict()

		self.sb = superblock(self)
		self.root = self.geti(self.ROOTIN)

	def geti(self, ino):
		if ino not in self.icache:
			self.icache[ino] = inode(self, ino)
			self.icache[ino].init()
		return self.icache[ino]
		
	def bread(self, b):
		x = b * self.BSIZE
		return self.img[x : x + self.BSIZE]

	def show(self, fo):
		self.root.render(".", "", fo)

if True:
	import glob

	for fn in glob.glob("Vol_*.bin"):
		c = coherent_fs(fn)
		fo = open(fn + ".txt", "w")
		c.show(fo)
		fo.close
